# A generic object that can be written on.

package Notepad;
use strict;
use vars qw(@ISA);
use Thing;
use Verb;
use UNIVERSAL qw(isa);
use Text;
use Error;
@ISA=qw{Thing};

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $this  = Thing::new($class,
		text => [],
		@_);
	bless ($this, $class);
	return $this;
}

# Tell if there is writing on the notepad, and approximatly how much.
sub look {
	my $this=shift;
	
	my $text=$this->text;
	my $size=0;
	my $d="";
	if (ref($text) eq 'ARRAY') {
		$size=length(join '', @$text);
	}		
	if ($size > 256) {
		$d="There is a lot of writing on it.";
	}
	elsif ($size > 128) {
		$d="There is some writing on it.";
	}
	elsif ($size > 0) {
		$d="There is a little writing on it.";
	}
	else {
		$d="It is blank.";
	}
	
	return ($this->description,$d);
}

# Show the text in a notepad. If it's called as "list", it shows line numbers
# too.
# TODO: range support.
sub verb_read {
	my $this=shift;
	my $verbcall=shift;

	my $text=$this->text;
	if (Error::iserror($text)) {
		# Although, there should be a good reason why not, in the
		# virtual world. Most paper can be read and written unless
		# it's locked up..
		return Error->new("You arn't allowed to read ".$this->name.".");
	}

	# This really shouldn't happen, but it's worth being safe.
	if (! $text) {
		$text=[];
	}

	if ($verbcall->word('verb') eq 'list') {
		# Figure out how long the longest number in the list will
		# be so we can pad.
		my $pad=length($#$text + 2)+1;
		my $pos=$this->pos;
		if ($pos == 0) {
			# Zero means at end.
			$pos=$#$text + 2;
		}
		elsif ($pos > $#$text + 2) {
			# If overshot, correct.
			$pos=$#$text + 2;
		}
		my $c=1;
		return map((($c == $pos) ? "->" : "  ") . sprintf("%${pad}i.  %s",$c++, $_),
			@$text, '');
	}
	else {
		return "It is blank." unless @$text;
		return $this->name." has the following written on it: ",
			map("  ".$_, @$text);
			
	}
}

# Add to what's written on the notepad, at the insert position.
sub verb_write {
	my $this=shift;
	my $verbcall=shift;
	
	my $text=$this->text;
	if (Error::iserror($text)) {
		return Error->new("You can't even read, much less write to ".$this->name.".");
	}
	
	my $pos=$this->pos;
	if ($pos == 0) {
		# Zero means at end.
		$pos=$#$text;
	}
	elsif ($pos > $#$text) {
		# If overshot, correct.
		$pos=$#$text;
	}
	
	@$text=(@$text[0..$pos], $verbcall->word('direct_object'), @$text[$pos + 1 .. $#$text]);
	my $ret=$this->text($text);
	if (Error::iserror($ret)) {
		return Error->new("You can't write to ".$this->name.".");
	}
	else {
		$verbcall->caller->location->announce($verbcall->caller->name." writes on ".$this->name.".");
		return "You write on ".$this->name.".";
	}
}

# Remove a line.
# TODO: range support
sub verb_erase {
	my $this=shift;
	my $verbcall=shift;
	
	my $text=$this->text;
	if (Error::iserror($text)) {
		return Error->new("You can't even read, much less erase lines from ".$this->name.".");
	}
	
	my $line=$verbcall->word('preposition');
	if (int($line) < 1 || int($line) > $#{$text} + 1) {
		return Error->new("No such line.");
	}

	@$text=(@$text[0..$line - 2], @$text[$line..$#$text]);
	my $ret=$this->text($text);
	if (Error::iserror($ret)) {
		return Error->new("You can't erase lines from ".$this->name.".");
	}		
	else {
		$verbcall->caller->location->announce($verbcall->caller->name." erases something from ".$this->name.".");
		return "Line $line erased.";
	}
}

1
